% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [fields] = FindStationary(intFunc,initCond,opt,varargin)
% FINDSTATIONARY returns the equilibrium value of the director field.  It
% integrates the given set of equations starting from the given initial
% condition until a stationary solution is reached, i.e. until the
% variations between two successive outputs is smaller than a given
% threshold.
%
% The algorithm is as follows:
% 0) Set the initial conditions as the starting point of the integration.
% 1) Integrate the equations from the starting point of the integration for
%    a given time tInt.
% 2) Compare the new solution with the starting point.
% 3) If the difference between the new and old value is sufficiently small
%    then exit and return the new value as the stationary value.
% 4) Otherwise increase tInt by a given factor (differences become
%    exponentially small as we reach equilibrium and therefore the
%    integration time must increase exponentially for significant
%    changes of the solutions to appear), set the new solution as starting
%    point and repeat the loop from point (1).
% 
% This algorithm may give rise to an infinite loop if a stationary solution
% is never reached.   Therefore the is an exit clause: if the number of
% iterations exceeds a certain threshold the program exits with an error
% message.
%
% [fields] = FindStationary(@DirectorFunction,initCond,opt,[parameters])
% The first argument is a handle to a function that integrates a given set
% of director equations for a certain time.     The second argument
% (initCond) is an array of initial conditions.   Its nature depends on the
% function specified in the first argument.
%
% The call format of the @DirectorField function must be:
%      @DirectorFunction(tInt,initCond,<parameters>)
% where tInt is the integration time, initCond is an array of initial
% conditions as the one passed to FindStationary and <parameters>
% represents all the parameters needed by the model to be integrated.
% 
% opt, if present, is a three argument array:
% opt(1) is the maximum percentage difference of the L1 norm of two
% successive fields for them to be considered stationary (default=0.001);
% opt(2) is the maximum number of iterations that the solver should take to
% reach a stationary solution.  If no solution is reached in opt(2)
% iterations the program reports an error (default=20).
% opt(3) is the initial integration time (default=1).
% opt(4) is the factor by which the integration time is increased at each
% interation (default sqrt(2)).
%
% Any additional parameter is passed to the integration routine.  If
% parameters must be passed to the integration routine, but there is no
% need to change the default option then replace opt with [].

% If the options have not been specified then give them the default values.
% The following  rather baroque construction is needed for backward
% compatibility with older versions of Matlab.  In Matlab 7 one would use
% the statement:
% if ((nargin==2) || (sum(size(opt)) ==0))
%     opt = [0.001,20,1,sqrt(2)];
% end
optDefault = [0.001,20,1,sqrt(2)];
if (nargin==2) 
    opt = optDefault;
else if (sum(size(opt)) ==0) 
    opt = optDefault;
    end 
end
% If the function name is passed as a string then convert it to a function
% handle.
if ischar(intFunc)
    disp 'converting function string to function handle ...'
    fieldsOde = str2func(intFunc);
else
    fieldsOde = intFunc;
end
% Start the integration loop.   At each iteration compare the new with the
% old solution.  Repeat the loop only if the difference is too large.  At
% each iteration increase the integration time by a given factor.
done = false;
tInt = opt(3);
oldFields = initCond;
iterCounter = 0;
while ~done
    newFields = feval(fieldsOde,tInt,oldFields,varargin{:});
    variation = sum(reshape(abs(newFields-oldFields),[1 numel(newFields)]))/ ...
                sum(reshape(abs(oldFields),[1 numel(newFields)]));
    if ( variation < opt(1))
        done = true;
    end
    iterCounter = iterCounter + 1;
    if ((~done) && (iterCounter >= opt(2)))
        warning('FindStationary:MaxIterExceeded',...
                'FindStationary - maximum number of iterations exceeded.');
        done = true;
    end
    tInt = tInt*opt(4);
    oldFields = newFields;
    %fprintf(1,'%d %f %f\n',iterCounter,tInt,variation);
end
fields = newFields;